<?php
/**
 * Plugin Name: 👾 MAJ Plugins & Thèmes
 * Description: Liste uniquement les extensions et thèmes installés manuellement (hors WordPress.org) + action de mise à jour.
 * Author: STY
 * Version: 1.12.1
 * Requires at least: 5.5
 * Requires PHP: 7.4
 * Text Domain: maj-plugins-themes
 */

if ( ! defined('ABSPATH') ) exit;

if ( ! class_exists('\WP_Upgrader_Skin') ) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
}

class MPT_Stream_Skin extends \WP_Upgrader_Skin {
    public function header() {
        nocache_headers();
        @ini_set('output_buffering','off');
        @ini_set('zlib.output_compression','0');
        @ini_set('implicit_flush','1');
        while (ob_get_level()) { @ob_end_flush(); }
        @ob_implicit_flush(1);
        if (function_exists('apache_setenv')) { @apache_setenv('no-gzip','1'); }
        if (!headers_sent()) {
            header('X-Accel-Buffering: no');
            header('Content-Type: text/html; charset='.get_option('blog_charset'));
        }

        // padding AVANT le <pre> pour déclencher le flush sans remplir le log
        echo '<!doctype html><meta name="robots" content="noindex"><style>
                body{font:13px/1.4 -apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Arial}
                .ok{color:#155724;background:#d4edda;padding:6px 8px;margin:8px 0;border-radius:3px}
                .err{color:#721c24;background:#f8d7da;padding:6px 8px;margin:8px 0;border-radius:3px}
                pre{white-space:pre-wrap;margin:0}
              </style>';
        echo '<!-- '.str_repeat(' ', 4096).' -->';

        // log visible dès la première ligne
        echo '<pre id="mpt-log">';
        // auto-scroll doux pendant le flux
        echo '<script>window.mptScroll=setInterval(function(){window.scrollTo(0,document.body.scrollHeight);},200);</script>';
        flush();
    }
    public function footer(){
        echo "</pre>";
        flush();
        echo '<script>try{clearInterval(window.mptScroll);}catch(e){};try{parent.postMessage({mptDone:true,mptOk:!!(window.mptOk),mptMsg:(window.mptMsg||"")}, "*")}catch(e){}</script>';
    }
    public function feedback($string, ...$args) {
        if ($string === '' || $string === null) return;

        // Remap des clés Core vers les messages humains
        if (is_string($string) && isset($this->upgrader, $this->upgrader->strings[$string])) {
            $string = $this->upgrader->strings[$string];
        }

        // Résolution des placeholders éventuels
        $msg = is_string($string) ? ($args ? @vsprintf($string, $args) : $string)
                                  : print_r($string, true);

        // Autoriser un HTML minimal utilisé par WP_Upgrader
        $allowed = [
            'a'    => ['href'=>true,'rel'=>true,'target'=>true],
            'span' => ['class'=>true],
            'code' => [],
            'strong'=>[], 'em'=>[], 'b'=>[], 'i'=>[], 'br'=>[]
        ];

        echo wp_kses((string)$msg, $allowed) . "\n"; // pas d'escape brut
        flush();
    }

    public function error($e){
        $t = is_wp_error($e) ? $e->get_error_message() : (string)$e;
        echo '<div class="err">'.wp_kses_post($t)."</div>\n";
        flush();
    }
    public function after(){}
}

final class Maj_Plugins_Themes {
    private $remote_plugins = [];  // slug => ['version'=>..., 'download'=>...]
    private $remote_themes  = [];  // slug => ['version'=>..., 'download'=>...]
    private $json_url       = 'https://ptn.stynet.com/index.json';
    private $current_theme_slug = '';
    private $repack_theme = false;
    private $index_live_ok = false;

    public function __construct() {
        add_action('admin_menu', [$this, 'menu']);
        add_action('admin_post_mpt_update', [$this, 'handle_update']);
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), [$this, 'add_settings_link']);
        add_filter('network_admin_plugin_action_links_' . plugin_basename(__FILE__), [$this, 'add_settings_link']);
    }

    public function menu() {
        add_management_page(
            __('👾 MAJ Plugins & Thèmes', 'maj-plugins-themes'),
            __('👾 MAJ Plugins & Thèmes', 'maj-plugins-themes'),
            'activate_plugins',
            'maj-plugins-themes',
            [$this, 'render_page']
        );
    }

    private function normalize_version(string $v): string {
        // trim + minuscules
        $v = strtolower(trim($v));
        if ($v === '' || $v === '—') return '';

        // retire préfixes courants
        $v = preg_replace('/^(version|ver|v)\s*/', '', $v);   // ex: "Version 2.3" → "2.3"

        // virgules → points
        $v = str_replace(',', '.', $v);

        // supprime espaces internes
        $v = preg_replace('/\s+/', '', $v);

        // garde uniquement chiffres, lettres et séparateurs .-+
        $v = preg_replace('/[^0-9a-z\.\-\+]/', '', $v);

        // pad numérique de tête à 3 segments: 2 → 2.0.0, 2.3 → 2.3.0
        if (preg_match('/^[0-9][0-9\.]*/', $v, $m)) {
            $head = $m[0];
            $rest = substr($v, strlen($head)); // suffixe pré-release éventuel
            $parts = array_map('intval', array_filter(explode('.', $head), 'strlen'));
            while (count($parts) < 3) $parts[] = 0;
            $v = implode('.', array_slice($parts, 0, 3)) . $rest;
        }
        return $v;
    }

    private function compare_versions(string $a, string $b): int {
        $na = $this->normalize_version($a);
        $nb = $this->normalize_version($b);
        if ($na === '' || $nb === '') return 0; // indécidable → égal
        return version_compare($na, $nb); // -1, 0, 1
    }


    private function bool_badge($on) {
        $class = $on ? 'badge-on' : 'badge-off';
        $label = $on ? __('Oui', 'maj-plugins-themes') : __('Non', 'maj-plugins-themes');
        return '<span class="'.$class.'">'.$label.'</span>';
    }

    private function is_plugin_dotorg($plugin_file, $data) {
        $t = get_site_transient('update_plugins');
        if ( ! $t || empty($t->last_checked) ) {
            if ( function_exists('wp_update_plugins') ) {
                wp_update_plugins();
                $t = get_site_transient('update_plugins');
            }
        }
        $entry = null;
        if ( is_object($t) ) {
            if ( isset($t->response[$plugin_file]) )       $entry = $t->response[$plugin_file];
            elseif ( isset($t->no_update[$plugin_file]) )  $entry = $t->no_update[$plugin_file];
        }
        if ( $entry ) {
            if ( !empty($entry->id) && strpos($entry->id, 'w.org/plugins/') === 0 ) return true;
            if ( !empty($entry->url) && strpos($entry->url, 'wordpress.org/plugins/') !== false ) return true;
            if ( !empty($entry->package) && strpos($entry->package, 'downloads.wordpress.org') !== false ) return true;
        } else {
            $update_uri = '';
            if ( isset($data['UpdateURI']) ) $update_uri = trim((string)$data['UpdateURI']);
            if ( isset($data['Update URI']) && $update_uri === '' ) $update_uri = trim((string)$data['Update URI']);
            if ( $update_uri ) {
                if ( $update_uri === 'wordpress.org' || strpos($update_uri, 'wordpress.org') !== false ) return true;
            } else {
                $plugin_uri = isset($data['PluginURI']) ? (string)$data['PluginURI'] : '';
                if ( $plugin_uri && strpos($plugin_uri, 'wordpress.org/plugins/') !== false ) return true;
            }
        }
        return false;
    }

    private function is_theme_dotorg($stylesheet, WP_Theme $theme) {
        $t = get_site_transient('update_themes');
        if ( ! $t || empty($t->last_checked) ) {
            if ( function_exists('wp_update_themes') ) {
                wp_update_themes();
                $t = get_site_transient('update_themes');
            }
        }
        $entry = null;
        if ( is_object($t) ) {
            if ( isset($t->response[$stylesheet]) )       $entry = (object) $t->response[$stylesheet];
            elseif ( isset($t->no_update[$stylesheet]) )  $entry = (object) $t->no_update[$stylesheet];
        }
        if ( $entry ) {
            if ( !empty($entry->url) && strpos($entry->url, 'wordpress.org/themes/') !== false ) return true;
            if ( !empty($entry->package) && strpos($entry->package, 'downloads.wordpress.org') !== false ) return true;
        } else {
            $update_uri = trim((string) $theme->get('UpdateURI'));
            if ( $update_uri ) {
                if ( $update_uri === 'wordpress.org' || strpos($update_uri, 'wordpress.org') !== false ) return true;
            } else {
                $theme_uri = (string) $theme->get('ThemeURI');
                if ( $theme_uri && strpos($theme_uri, 'wordpress.org/themes/') !== false ) return true;
            }
        }
        return false;
    }

    private function load_index_versions(): void {
        $this->index_live_ok  = false;
        $this->remote_plugins = [];
        $this->remote_themes  = [];

        $resp = wp_remote_get($this->json_url, [
            'timeout' => 7,
            'redirection' => 3,
            'reject_unsafe_urls' => true,
        ]);
        if (is_wp_error($resp)) return;
        if ((int) wp_remote_retrieve_response_code($resp) !== 200) return;

        $body = wp_remote_retrieve_body($resp);
        $json = json_decode($body, true);
        if (!is_array($json)) return;

        $base    = trailingslashit(dirname($this->json_url));
        $plugins = [];
        $themes  = [];

        if (!empty($json['plugins']) && is_array($json['plugins'])) {
            foreach ($json['plugins'] as $it) {
                $slug = (string)($it['slug'] ?? '');
                $ver  = (string)($it['version'] ?? '');
                $file = ltrim((string)($it['file'] ?? ''), '/');
                $dl   = ($slug !== '' && $file !== '') ? $base.$file : '';
                if ($slug !== '' && $ver !== '') $plugins[$slug] = ['version'=>$ver,'download'=>$dl];
            }
        }
        if (!empty($json['themes']) && is_array($json['themes'])) {
            foreach ($json['themes'] as $it) {
                $slug = (string)($it['slug'] ?? '');
                $ver  = (string)($it['version'] ?? '');
                $file = ltrim((string)($it['file'] ?? ''), '/');
                $dl   = ($slug !== '' && $file !== '') ? $base.$file : '';
                if ($slug !== '' && $ver !== '') $themes[$slug] = ['version'=>$ver,'download'=>$dl];
            }
        }

        $this->remote_plugins = $plugins;
        $this->remote_themes  = $themes;
        $this->index_live_ok  = true;
    }

    private function td_remote_version(string $remote, string $local): string {
        if ($remote === '') $remote = '—';
        $highlight = ($remote !== '—' && $local !== '—' && $this->compare_versions($remote, $local) === 1);
        $style = $highlight ? ' style="background:yellow; padding:5px; font-weight:bold;"' : '';
        return '<td class="mono"><span'.$style.'>'.esc_html($remote).'</span></td>';
    }

    private function action_button(string $type, string $slug, string $local, string $remote, string $download): string {
        $label = esc_html__('Mettre à jour', 'maj-plugins-themes');

        if (!$this->index_live_ok) {
            return '<td><button class="button" disabled title="'.esc_attr__('Index indisponible. Veuillez réessayer plus tard.', 'maj-plugins-themes').'">'.$label.'</button></td>';
        }

        $can_update = ($remote !== '' && $local !== '—' && $this->compare_versions($remote, $local) === 1 && $download !== '');
        if (!$can_update) return '<td><button class="button" disabled>'.$label.'</button></td>';

        $url = admin_url('admin-post.php');
        $nonce = wp_create_nonce('mpt_update_'.$type.'_'.$slug.'_'.$remote);
        $html  = '<form method="post" action="'.esc_url($url).'" target="mpt-runner">';
        $html .= '<input type="hidden" name="mpt_mode" value="stream">'; 
        $html .= '<input type="hidden" name="action" value="mpt_update">';
        $html .= '<input type="hidden" name="type" value="'.esc_attr($type).'">';
        $html .= '<input type="hidden" name="slug" value="'.esc_attr($slug).'">';
        $html .= '<input type="hidden" name="version" value="'.esc_attr($remote).'">';
        $html .= '<input type="hidden" name="download" value="'.esc_attr($download).'">';
        $html .= '<input type="hidden" name="_wpnonce" value="'.esc_attr($nonce).'">';
        $html .= '<button class="button button-primary">'.$label.'</button>';
        $html .= '</form>';
        return '<td>'.$html.'</td>';
    }

    /* === Helpers ZIP: détecter parent vs child, repackage, renommer dossier === */

    private function zip_find_style_path(\ZipArchive $zip): string {
        $preferred = '';
        for ($i=0; $i<$zip->numFiles; $i++) {
            $name = $zip->getNameIndex($i);
            if (!is_string($name)) continue;
            if (preg_match('#^([^/]+)/style\.css$#i', $name, $m)) {
                if ($this->current_theme_slug && strtolower($m[1]) === strtolower($this->current_theme_slug)) return $name;
                if ($preferred === '') $preferred = $name;
            } elseif (strcasecmp($name, 'style.css') === 0 && $preferred === '') {
                $preferred = $name;
            }
        }
        return $preferred;
    }

    private function zip_is_child_theme(\ZipArchive $zip): bool {
        $stylePath = $this->zip_find_style_path($zip);
        if ($stylePath === '') return false;
        $css = $zip->getFromName($stylePath);
        if (!is_string($css)) return false;
        return (bool) preg_match('/^\s*Template\s*:\s*(.+)$/mi', $css);
    }

    private function zip_has_theme_root(\ZipArchive $zip, string $expect_slug = ''): bool {
        $found_any = false;
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $name = $zip->getNameIndex($i);
            if (!is_string($name)) continue;
            if (preg_match('#^([^/]+)/style\.css$#i', $name, $m)) {
                $found_any = true;
                if ($expect_slug === '' || strtolower($m[1]) === strtolower($expect_slug)) return true;
            }
            if (strcasecmp($name, 'style.css') === 0) {
                $found_any = true;
                if ($expect_slug === '') return true;
            }
        }
        return ($expect_slug === '') ? $found_any : false;
    }

    private function extract_inner_zip_to_tmp(\ZipArchive $outer, string $entry): string {
        $stream = $outer->getStream($entry);
        if (!$stream) return '';
        $tmp = wp_tempnam(basename($entry) ?: 'inner.zip');
        if (!$tmp) { fclose($stream); return ''; }
        $fh = fopen($tmp, 'w');
        if (!$fh) { fclose($stream); return ''; }
        while (!feof($stream)) {
            $buf = fread($stream, 1048576);
            if ($buf === false) break;
            fwrite($fh, $buf);
        }
        fclose($stream);
        fclose($fh);
        return $tmp;
    }

    public function add_settings_link(array $links): array {
        $url = admin_url('tools.php?page=maj-plugins-themes');
        array_unshift($links, '<a href="'.esc_url($url).'">'.esc_html__('Réglages', 'maj-plugins-themes').'</a>');
        return $links;
    }

    public function maybe_repackage_theme($reply, $package, $upgrader, $hook_extra) {
        if (!$this->repack_theme || !class_exists('\ZipArchive')) return $reply;

        $bundle = download_url($package);
        if (is_wp_error($bundle)) return $reply;

        $zip = new \ZipArchive();
        if ($zip->open($bundle) !== true) { @unlink($bundle); return $reply; }

        if ($this->zip_has_theme_root($zip, $this->current_theme_slug) && ! $this->zip_is_child_theme($zip)) {
            $zip->close();
            return $bundle; // déjà un parent installable
        }

        $candidates = [];
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $name = $zip->getNameIndex($i);
            if (!is_string($name) || !preg_match('#\.zip$#i', $name)) continue;
            if (preg_match('#child#i', $name)) continue; // skip child évidents
            $prio = 2;
            $basename = strtolower(basename($name));
            $slug = strtolower($this->current_theme_slug);
            if ($basename === $slug.'.zip') $prio = 0;
            elseif (strpos($basename, $slug) === 0) $prio = 1;
            $candidates[] = [$prio, $name];
        }
        usort($candidates, function($a,$b){ return $a[0] <=> $b[0]; });

        foreach ($candidates as [, $entry]) {
            $inner = $this->extract_inner_zip_to_tmp($zip, $entry);
            if ($inner === '') continue;
            $iz = new \ZipArchive();
            if ($iz->open($inner) === true) {
                $installable = $this->zip_has_theme_root($iz, $this->current_theme_slug);
                $is_child    = $this->zip_is_child_theme($iz);
                $iz->close();
                if ($installable && ! $is_child) {
                    $zip->close();
                    @unlink($bundle);
                    return $inner; // parent trouvé
                }
            }
            @unlink($inner);
        }

        $zip->close();
        return $bundle; // fallback
    }

    public function fix_theme_root_folder($source, $remote_source, $upgrader, $hook_extra) {
        if (!$this->repack_theme) return $source;
        $slug = $this->current_theme_slug ?: '';
        if ($slug === '') return $source;

        $src = untrailingslashit($source);

        // style.css à la racine
        if (file_exists($src.'/style.css')) {
            $css = @file_get_contents($src.'/style.css');
            if (is_string($css) && preg_match('/^\s*Template\s*:\s*(.+)$/mi', $css)) {
                return trailingslashit($src); // child → ne pas toucher
            }
            $base = basename($src);
            if ($base !== $slug) {
                $new = trailingslashit(dirname($src)).$slug;
                @rename($src, $new);
                return trailingslashit($new);
            }
            return trailingslashit($src);
        }

        // Sous-dossier contenant style.css non-child
        $best = '';
        if (is_dir($src) && ($dh = opendir($src))) {
            while (($entry = readdir($dh)) !== false) {
                if ($entry === '.' || $entry === '..') continue;
                $p = $src.'/'.$entry;
                if (is_dir($p) && file_exists($p.'/style.css')) {
                    $css = @file_get_contents($p.'/style.css');
                    if (is_string($css) && preg_match('/^\s*Template\s*:\s*(.+)$/mi', $css)) {
                        continue;
                    }
                    $best = $p; break;
                }
            }
            closedir($dh);
        }
        if ($best !== '') {
            $base = basename($best);
            if ($base !== $slug) {
                $new = trailingslashit(dirname($best)).$slug;
                @rename($best, $new);
                return trailingslashit($new);
            }
            return trailingslashit($best);
        }

        return $source;
    }

    public function handle_update() {
        $this->load_index_versions();
        if (!$this->index_live_ok) wp_die(__('Index indisponible. Veuillez réessayer plus tard.', 'maj-plugins-themes'));

        $in_stream = (($_POST['mpt_mode'] ?? '') === 'stream');
        // fallback si le navigateur indique une soumission vers un iframe (au cas où le WAF altère le POST)
        if (!$in_stream && isset($_SERVER['HTTP_SEC_FETCH_DEST']) && $_SERVER['HTTP_SEC_FETCH_DEST'] === 'iframe') {
         $in_stream = true;
        }
        $skin = $in_stream ? new \MPT_Stream_Skin() : new \Automatic_Upgrader_Skin();
        if ( $in_stream && ! defined('IFRAME_REQUEST') ) define('IFRAME_REQUEST', true);

        if ( ! current_user_can('install_plugins') ) wp_die(__('Permissions insuffisantes.', 'maj-plugins-themes'));

        $type     = isset($_POST['type']) ? sanitize_key($_POST['type']) : '';
        $slug     = isset($_POST['slug']) ? sanitize_text_field($_POST['slug']) : '';
        $version  = isset($_POST['version']) ? sanitize_text_field($_POST['version']) : '';
        $download = isset($_POST['download']) ? esc_url_raw($_POST['download']) : '';
        if ( ! wp_verify_nonce($_POST['_wpnonce'] ?? '', 'mpt_update_'.$type.'_'.$slug.'_'.$version) ) wp_die(__('Échec nonce.', 'maj-plugins-themes'));
        if ( $type === '' || $slug === '' || $version === '' || $download === '' ) wp_die(__('Paramètres manquants.', 'maj-plugins-themes'));

        require_once ABSPATH.'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH.'wp-admin/includes/file.php';
        require_once ABSPATH.'wp-admin/includes/plugin.php';
        require_once ABSPATH.'wp-admin/includes/theme.php';

        // --- début flux iframe
        if ($in_stream && method_exists($skin,'header')) $skin->header();
        if ($in_stream && method_exists($skin,'feedback')) $skin->feedback(sprintf('→ %s %s …', $type==='plugin'?'Plugin':'Thème', $slug));

        if ( $type === 'plugin' ) {
            $upgrader = new \Plugin_Upgrader($skin);
            $res = $upgrader->install($download, ['overwrite_package'=>true]);
            $ok  = ($res === true);
            $msg = $ok ? 'Plugin mis à jour.' : 'Échec mise à jour plugin.';
        } elseif ( $type === 'theme' ) {
            $this->current_theme_slug = $slug;
            $this->repack_theme = true;
            add_filter('upgrader_pre_download',     [$this,'maybe_repackage_theme'],10,4);
            add_filter('upgrader_source_selection', [$this,'fix_theme_root_folder'],10,4);

            $upgrader = new \Theme_Upgrader($skin);
            $res = $upgrader->install($download, ['overwrite_package'=>true,'clear_destination'=>true,'clear_working'=>true]);

            remove_filter('upgrader_pre_download',     [$this,'maybe_repackage_theme'],10);
            remove_filter('upgrader_source_selection', [$this,'fix_theme_root_folder'],10);
            $this->repack_theme = false;

            $ok  = ($res === true);
            if ($ok) { if (function_exists('wp_clean_themes_cache')) wp_clean_themes_cache(true); delete_site_transient('update_themes'); }
            $msg = $ok ? 'Thème mis à jour.' : 'Échec mise à jour thème.';
        } else {
            wp_die(__('Type invalide.', 'maj-plugins-themes'));
            return;
        }

        if ($in_stream && method_exists($skin,'feedback')) $skin->feedback($msg);
        if ($in_stream) {
            printf('<script>window.mptOk=%s;window.mptMsg=%s;</script>',
                $ok ? 'true' : 'false',
                json_encode($msg)
            );
        }
        if ($in_stream && method_exists($skin,'footer'))   $skin->footer();
        if ($in_stream) exit; // ne jamais rediriger depuis l’iframe
        // --- fin flux iframe

        $redirect = add_query_arg(['page'=>'maj-plugins-themes','mpt_msg'=>rawurlencode($msg),'mpt_ok'=>$ok?'1':'0'], admin_url('tools.php'));
        wp_safe_redirect($redirect);
        exit;
    }

    public function render_page() {
        $this->load_index_versions();
        $live_ok = $this->index_live_ok;

        $auto_plugins = (array) get_site_option('auto_update_plugins', []);
        $auto_themes  = (array) get_site_option('auto_update_themes', []);

        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        $plugins = get_plugins();

        $themes = wp_get_themes();
        $active_theme_stylesheet = wp_get_theme()->get_stylesheet();

        // === POP-UP d'alerte MAJ pour "maj-plugins-themes" ===
        $self_file  = plugin_basename(__FILE__);                  // ex: maj-plugins-themes/maj-plugins-themes.php
        $self_slug  = dirname($self_file);                        // ex: maj-plugins-themes
        $self_local = isset($plugins[$self_file]['Version']) ? (string)$plugins[$self_file]['Version'] : '';
        $self_remote = $live_ok ? (string)($this->remote_plugins[$self_slug]['version'] ?? '') : '';
        $self_download = $live_ok ? (string)($this->remote_plugins[$self_slug]['download'] ?? '') : '';
        $self_needs_update = ($self_remote !== '' && $self_local !== '' && $this->compare_versions($self_remote, $self_local) === 1 && $self_download !== '');

        if ($self_needs_update) {
            $nonce = wp_create_nonce('mpt_update_plugin_'.$self_slug.'_'.$self_remote);

            // Pop-up d'information
            echo '<style>
                #mpt-info{position:fixed;inset:0;background:rgba(0,0,0,.35);z-index:100002;display:none}
                #mpt-info .box{position:absolute;top:12%;left:50%;transform:translateX(-50%);
                  width:min(760px,92vw);background:#fff;border:1px solid #ccd0d4;border-radius:6px;
                  box-shadow:0 10px 30px rgba(0,0,0,.2);display:flex;flex-direction:column}
                #mpt-info .hdr{padding:14px 16px;border-bottom:1px solid #e2e4e7;font-weight:600}
                #mpt-info .cnt{padding:14px 16px;line-height:1.5}
                #mpt-info .ftr{display:flex;justify-content:flex-end;gap:8px;padding:12px 16px;border-top:1px solid #e2e4e7}
            </style>';

            echo '<div id="mpt-info" role="dialog" aria-modal="true" aria-label="Information mise à jour">
                    <div class="box">
                        <div class="hdr">Information</div>
                        <div class="cnt">
                            <p><strong>Une nouvelle mise à jour est disponible pour le plugin MAJ Plugins et Thèmes.</strong></p>
                            <p>Il est fortement conseillé d\'effectuer cette mise à jour avant de mettre à jour les autres plugins.</p>
                            <p class="mono subtle">Actuelle : '.esc_html($self_local).' &nbsp;→&nbsp; Disponible : <strong>'.esc_html($self_remote).'</strong></p>
                        </div>
                        <div class="ftr">
                            <button type="button" class="button" id="mpt-info-close">Fermer</button>
                            <form id="mpt-info-update" method="post" action="'.esc_url(admin_url('admin-post.php')).'">
                                <input type="hidden" name="mpt_mode" value="stream">
                                <input type="hidden" name="action" value="mpt_update">
                                <input type="hidden" name="type" value="plugin">
                                <input type="hidden" name="slug" value="'.esc_attr($self_slug).'">
                                <input type="hidden" name="version" value="'.esc_attr($self_remote).'">
                                <input type="hidden" name="download" value="'.esc_attr($self_download).'">
                                <input type="hidden" name="_wpnonce" value="'.esc_attr($nonce).'">
                                <button class="button button-primary" id="mpt-info-run">Mettre à jour maintenant</button>
                            </form>
                        </div>
                    </div>
                  </div>';

            echo '<script>
                (function(){
                  var info   = document.getElementById("mpt-info");
                  var closeB = document.getElementById("mpt-info-close");
                  var form   = document.getElementById("mpt-info-update");
                  var runner = document.getElementById("mpt-runner");     // iframe déjà présent dans la page
                  var modal  = document.getElementById("mpt-modal");       // modal de logs existant
                  function show(el){ if(el) el.style.display="block"; }
                  function hide(el){ if(el) el.style.display="none"; }
                  document.addEventListener("DOMContentLoaded", function(){ show(info); });
                  if(closeB){ closeB.addEventListener("click", function(){ hide(info); }); }
                  if(form){
                    form.addEventListener("submit", function(e){
                      // route la soumission vers l’iframe et ouvre le modal de logs existant
                      if(form.getAttribute("target") !== "mpt-runner") form.setAttribute("target","mpt-runner");
                      show(modal);
                      hide(info);
                    }, {once:true});
                  }
                })();
            </script>';
        }
        // === FIN pop-up d'alerte ===


        echo '<style>
            .widefat .column-status{width:140px}
            .widefat .column-version{width:140px}
            .widefat .column-action{width:140px}
            .badge-on{background:#46b450;color:#fff;padding:2px 8px;border-radius:12px}
            .badge-off{background:#dc3232;color:#fff;padding:2px 8px;border-radius:12px}
            .pill{display:inline-block;padding:2px 8px;border-radius:12px;background:#e5e5e5}
            .subtle{color:#555}
            .muted{color:#777;font-style:italic}
            .mono{font-family:Menlo,Consolas,monospace}
            .below-link{margin:0}
            .redcolor{color:red; }
        </style>';

        echo '<style>
        #mpt-modal{position:fixed;inset:0;background:rgba(0,0,0,.35);z-index:100001;display:none}
        #mpt-modal .box{position:absolute;top:8%;left:50%;transform:translateX(-50%);
          width:min(1000px,92vw);background:#fff;border:1px solid #ccd0d4;border-radius:6px;
          box-shadow:0 10px 30px rgba(0,0,0,.2);display:flex;flex-direction:column}
        #mpt-modal .hdr{padding:12px 16px;border-bottom:1px solid #e2e4e7;font-weight:600}
        #mpt-modal .log{height:360px;overflow:auto}
        #mpt-modal iframe{width:100%;height:100%;border:0;display:block}
        #mpt-modal .ftr{display:flex;justify-content:flex-end;gap:8px;padding:12px 16px;border-top:1px solid #e2e4e7}
        </style>';

        echo '<div class="wrap"><h1>'.esc_html__('Mise à jour des Plugins & Thèmes', 'maj-plugins-themes').'</h1>';

        // === Vérification en-tête du serveur de mise à jour ===
        $gateway_warning_lines = [];

        $head_resp = wp_remote_head('https://ptn.stynet.com/', [
            'timeout'            => 5,
            'redirection'        => 1,
            'reject_unsafe_urls' => true,
        ]);

        if (is_wp_error($head_resp)) {
            $err_msg = $head_resp->get_error_message();
            if ($err_msg === '') {
                $err_msg = __('erreur inconnue', 'maj-plugins-themes');
            }
            $gateway_warning_lines[] = sprintf(
                __('L’accès au serveur de mise à jour a échoué : %s', 'maj-plugins-themes'),
                $err_msg
            );
        } else {
            $code = (int) wp_remote_retrieve_response_code($head_resp);
            if ($code !== 200) {
                $label = trim(wp_remote_retrieve_response_message($head_resp));
                if ($code === 404) {
                    $label = __('page introuvable', 'maj-plugins-themes');
                } elseif ($code === 403) {
                    $label = __('accès refusé', 'maj-plugins-themes');
                } elseif ($label === '') {
                    $label = __('erreur inconnue', 'maj-plugins-themes');
                }

                $gateway_warning_lines[] = sprintf(
                    __('L’accès au serveur de mise à jour a renvoyé l’erreur : %d (%s).', 'maj-plugins-themes'),
                    $code,
                    $label
                );

            } else {
                $head_index_resp = wp_remote_head('https://ptn.stynet.com/index.json', [
                    'timeout'            => 5,
                    'redirection'        => 1,
                    'reject_unsafe_urls' => true,
                ]);
                // --- Si https://ptn.stynet.com/ est ok (code 200), mais que l’index n’est pas joignable,
                // afficher un message générique avec lien mailto ---
                if ( is_wp_error( $head_index_resp ) || (int) wp_remote_retrieve_response_code( $head_index_resp ) !== 200 ) {
                    $gateway_warning_lines[] = sprintf(
                        __(
                            'L\'index distant du catalogue des téléchargements n\'est actuellement pas disponible pour effectuer des mises à jour, veuillez réessayer plus tard&nbsp;! Si cela persiste, envoyez un mail sur %s.',
                            'maj-plugins-themes'
                        ),
                        '<a href="mailto:contact@stynet.com">contact@stynet.com</a>'
                    );
                }
            }
        }

        if (!empty($gateway_warning_lines)) {
            echo '<div class="notice notice-warning redcolor">';
            foreach ($gateway_warning_lines as $line) {
                echo '<p>'.$line.'</p>';
            }
            echo '</div>';
        }

        // === Fin vérification en-tête ===

        echo '
        <div id="mpt-modal" role="dialog" aria-modal="true" aria-label="Journal de mise à jour">
          <div class="box">
            <div class="hdr">'.esc_html__('Mise à jour en cours… Ne quitte pas cette page.', 'maj-plugins-themes').'</div>
            <div class="log"><iframe id="mpt-runner" name="mpt-runner"></iframe></div>
            <div class="ftr">
              <button type="button" class="button" id="mpt-close" disabled>Fermer</button>
            </div>
          </div>
        </div>';

        // On n’affiche le message générique index.json indisponible
        // que s’il n’y a pas déjà un warning spécifique sur le gateway.
        if (!$live_ok && empty($gateway_warning_lines)) {
            echo '<div class="notice notice-warning redcolor"><p>'
                . esc_html__('L’index distant (index.json) est indisponible ou illisible. Veuillez rafraîchir la page ou réessayer plus tard.', 'maj-plugins-themes')
                . '</p></div>';
        }

        if (isset($_GET['mpt_msg'])) {
            $ok = isset($_GET['mpt_ok']) && $_GET['mpt_ok'] === '1';
            $cls = $ok ? 'updated' : 'error';
            echo '<div class="'.$cls.' notice is-dismissible"><p>'.esc_html(wp_kses_post($_GET['mpt_msg'])).'</p></div>';
        }

        // === PLUGINS (manuels) ===
        echo '<h2>'.esc_html__('Extensions (installées manuellement)', 'maj-plugins-themes').'</h2>';
        echo '<table class="widefat striped"><thead><tr>
                <th>'.esc_html__('Nom', 'maj-plugins-themes').'</th>
                <th class="column-status">'.esc_html__('Actif', 'maj-plugins-themes').'</th>
                <th class="column-status">'.esc_html__('Auto-mises à jour', 'maj-plugins-themes').'</th>
                <th class="column-version">'.esc_html__('Version actuelle', 'maj-plugins-themes').'</th>
                <th class="column-version">'.esc_html__('Version disponible', 'maj-plugins-themes').'</th>
                <th class="column-action">'.esc_html__('Action', 'maj-plugins-themes').'</th>
            </tr></thead><tbody>';

        $manual_plugins = 0;
        foreach ($plugins as $plugin_file => $data) {
            if ( $this->is_plugin_dotorg($plugin_file, $data) ) continue;

            $manual_plugins++;
            $name    = isset($data['Name']) ? $data['Name'] : $plugin_file;
            $version = isset($data['Version']) && $data['Version'] !== '' ? $data['Version'] : '—';
            $is_active  = is_plugin_active($plugin_file);
            $is_network = function_exists('is_plugin_active_for_network') && is_plugin_active_for_network($plugin_file);
            $status_html  = $is_network ? '<span class="pill">Réseau</span>' : $this->bool_badge($is_active);
            $auto = in_array($plugin_file, $auto_plugins, true);

            $slug = dirname($plugin_file);
            $remote_ver = $live_ok ? ($this->remote_plugins[$slug]['version']  ?? '') : '';
            $download   = $live_ok ? ($this->remote_plugins[$slug]['download'] ?? '') : '';

            echo '<tr>
                    <td><strong>'.esc_html($name).'</strong><br><span class="subtle mono">'.esc_html($slug).'</span></td>
                    <td>'.$status_html.'</td>
                    <td>'.$this->bool_badge($auto).'</td>
                    <td class="mono">'.esc_html($version).'</td>'.
                    $this->td_remote_version($remote_ver, $version).
                    $this->action_button('plugin', $slug, $version, $remote_ver, $download).
                '</tr>';
        }
        if ( $manual_plugins === 0 ) {
            echo '<tr><td colspan="6"><span class="muted">'.esc_html__('Aucune extension installée manuellement détectée.', 'maj-plugins-themes').'</span></td></tr>';
        }
        echo '</tbody></table>';
        echo '<p class="below-link"><a href="'.esc_url( admin_url('plugins.php') ).'">'.esc_html__('Aller à Extensions', 'maj-plugins-themes').'</a></p>';

        // === THEMES (manuels) ===
        echo '<h2 style="margin-top:24px">'.esc_html__('Thèmes (installés manuellement)', 'maj-plugins-themes').'</h2>';
        echo '<table class="widefat striped"><thead><tr>
                <th>'.esc_html__('Nom', 'maj-plugins-themes').'</th>
                <th class="column-status">'.esc_html__('Actif', 'maj-plugins-themes').'</th>
                <th class="column-status">'.esc_html__('Auto-mises à jour', 'maj-plugins-themes').'</th>
                <th class="column-version">'.esc_html__('Version actuelle', 'maj-plugins-themes').'</th>
                <th class="column-version">'.esc_html__('Version disponible', 'maj-plugins-themes').'</th>
                <th class="column-action">'.esc_html__('Action', 'maj-plugins-themes').'</th>
            </tr></thead><tbody>';

        $manual_themes = 0;
        foreach ($themes as $stylesheet => $theme) {
            if ( $this->is_theme_dotorg($stylesheet, $theme) ) continue;
            if ( $theme->parent() ) continue; // ignore child themes

            $manual_themes++;
            $name    = $theme->get('Name') ?: $stylesheet;
            $version = $theme->get('Version') ?: '—';
            $is_active_theme = ($stylesheet === $active_theme_stylesheet);
            $auto = in_array($stylesheet, $auto_themes, true);

            $remote_ver = $live_ok ? ($this->remote_themes[$stylesheet]['version']  ?? '') : '';
            $download   = $live_ok ? ($this->remote_themes[$stylesheet]['download'] ?? '') : '';

            echo '<tr>
                    <td><strong>'.esc_html($name).'</strong><br><span class="subtle mono">'.esc_html($stylesheet).'</span></td>
                    <td>'.$this->bool_badge($is_active_theme).'</td>
                    <td>'.$this->bool_badge($auto).'</td>
                    <td class="mono">'.esc_html($version).'</td>'.
                    $this->td_remote_version($remote_ver, $version).
                    $this->action_button('theme', $stylesheet, $version, $remote_ver, $download).
                '</tr>';
        }
        if ( $manual_themes === 0 ) {
            echo '<tr><td colspan="6"><span class="muted">'.esc_html__('Aucun thème installé manuellement détecté.', 'maj-plugins-themes').'</span></td></tr>';
        }
        echo '</tbody></table>';
        echo '<p class="below-link"><a href="'.esc_url( admin_url('themes.php') ).'">'.esc_html__('Aller à Thèmes', 'maj-plugins-themes').'</a></p>';

        echo <<<HTML
        <script>
        (function(){
          function isUpdateForm(f){
            return f && f.tagName==="FORM" &&
                   f.querySelector('input[name="action"][value="mpt_update"]');
          }
          var modal   = document.getElementById("mpt-modal");
          var iframe  = document.getElementById("mpt-runner");
          var closeBt = document.getElementById("mpt-close");

          document.addEventListener("submit", function(e){
            var f = e.target; if(!isUpdateForm(f)) return;
            if(f.getAttribute("target") !== "mpt-runner") f.setAttribute("target","mpt-runner");
            if(modal) modal.style.display = "block";
            var btn = f.querySelector("button.button-primary");
            if(btn){ btn.disabled = true; btn.textContent = "Mise à jour…"; }
          }, true);

          // Fin de l’upgrade signalée par l’iframe
          window.addEventListener("message", function(ev) {
            if (ev && ev.data && ev.data.mptDone) {
              var hdr = modal.querySelector('.hdr');
              if (hdr) {
                hdr.textContent = ev.data.mptOk ? 'Mise à jour effectuée avec succès' : 'Échec de la mise à jour';
                hdr.style.color = ev.data.mptOk ? '#0a7d2f' : '#b32d2e';
              }
              if (closeBt) closeBt.disabled = false;
            }
          });

          if(closeBt){
            closeBt.addEventListener("click", function(){
              if(modal) modal.style.display = "none";
              var url = window.location.href.split("#")[0];
              window.location.replace(url);
            });
          }
        })();
        </script>
        HTML;

        echo '</div>';
    }


}

new Maj_Plugins_Themes();
